## Declared Functions

Declaration of functions written in MDX.




### Description

icCube allows for declaring a function within the scope of a `SELECT` query. The syntax is very much similar to the declaration of a calculated member.

<pre>
WITH FUNCTION fun(a,b) AS a + b
SELECT ...
</pre>

The parameter declaration supports some sort of `type declaration`. Currently the `Value` type is supported.

<pre>
WITH FUNCTION fun( Value val ) AS ....
</pre>

means that the `val` parameter will be converted to a value before being used within the function. When calling the function with a tuple `t`, this is equivalent to `fun( t.value )`. Declaring a parameter as a `Value` is simplifying the call of the method and ensure that the value is computed once only. E.g.: 
<pre>
WITH FUNCTION square( Value val ) AS val * val
</pre>

Note that a function can be passed as a `function parameter`.

A function can be declared as `static` using the `STATIC` keyword. In that case, the evaluation of the function is performed outside the current context and both the slicer and the sub-select are ignored. The optional `CACHED` property allows for caching the evaluation (default is true). Applies to `STATIC` or `CONST` functions only.

A `STATIC` function can have one or more parameters.

When declared at statement level and `CACHED`, it evaluated once per MDX execution.

When declared at schema level and `CACHED`, it is evaluated once.

<pre>
WITH STATIC FUNCTION fun() AS ... [, CACHED = true|false ]
SELECT ...
</pre>

A function can be declared as <b>const</b> using the `CONST` keyword. In that case, the evaluation of the function is considered as a constant and is evaluated once outside the current context except for the slicer and the sub-select. The optional CACHED property allows for caching the evaluation (default is true). Applies to `STATIC` or `CONST` functions only. Several limitations apply to const functions as described in the following examples.

A `CONST` function cannot have any parameter.

When declared at statement level and `CACHED`, it is evaluated once per MDX execution.

When declared at schema level and `CACHED`, it is evaluated once.

<pre>
WITH CONST FUNCTION fun() AS ... [, CACHED = true|false ]
SELECT ...
</pre>

Similarly, functions can be declared (and removed) at schema level using the `CREATE / DROP` statements.

<pre>
DROP FUNCTION fun
CREATE [STATIC|CONST] [NATIVE] FUNCTION fun(a,b) AS a+b, [ , CUBE = cube-name ] [, CACHED = yes|true ]
</pre>

<b>Native Functions</b>

A function can be declared as `native` using the `NATIVE` keyword. In that case, it is not defined using MDX. icCube supports Java integration. The following is demonstrating a native Java function:

<pre>
CREATE NATIVE FUNCTION RpJavaFormat( String locale, double amount, String curr ) AS
/* JAVA : String

    import crazydev.common.utils.CdLocaleUtils;

    return String.format(CdLocaleUtils.fromName(locale, Locale.getDefault()), "%,.2f", amount) + " " + curr;

*/
</pre>

Please refer to the [integration](../mdx_integration/index.md) section for more details.

The following examples are demonstrating several features of these function declarations.



             
### Examples

           

<div class="ic3-mdx-example-header ic3-mdx-example-header-collapsed" onclick="ic3_mdx_example_click(this)">

#### factorial

</div>

<div class="ic3-mdx-example-body" style="display:none">

In this example, we are defining the factorial of a number. You may notice the usage of a recursive notation.

**Query**

```
WITH
    FUNCTION factorial(n) AS Iif( n = 0, 1, n * factorial(n-1) )

    MEMBER f0 AS factorial(0)
    MEMBER f3 AS factorial(3)

SELECT { f0, f3 } ON 0 FROM [Sales]
```

 
**Result**

<table  class="ic3-mdx-result">
<tr>
  <td class="ic3-mdx-result-cell-header">f0</td>
  <td class="ic3-mdx-result-cell-header">f3</td>
</tr>
<tr>
 <td>1</td>
 <td>6</td>
</tr>
</table>



**Assertion : Cell Equals**

```
WITH
    MEMBER f0 AS 1
    MEMBER f3 AS 6

SELECT { f0, f3 } ON 0 FROM [Sales]
```


 
</div>

           

<div class="ic3-mdx-example-header ic3-mdx-example-header-collapsed" onclick="ic3_mdx_example_click(this)">

#### parameter typing - Value

</div>

<div class="ic3-mdx-example-body" style="display:none">

In this example, as we are declaring the parameter as a value, it is going to be evaluated before being used within the function. This makes the calling side simpler and ensure the tuple passed as parameter is evaluated once only.

**Query**

```
WITH
    FUNCTION square( Value t ) AS t * t

    MEMBER [Square Amount] AS square( [Measures].[Amount] )

SELECT [Square Amount] ON 0 FROM [Sales]
```

 
**Result**

<table  class="ic3-mdx-result">
<tr>
  <td class="ic3-mdx-result-cell-header">Square Amount</td>
</tr>
<tr>
 <td>1046529</td>
</tr>
</table>



**Assertion : Cell Equals**

```
WITH
    FUNCTION square( t ) AS t * t

    MEMBER [Square Amount] AS square( [Measures].[Amount].Value )

SELECT [Square Amount] ON 0 FROM [Sales]
```


**Assertion : Cell Equals**

```
WITH
    MEMBER [Square Amount] AS [Measures].[Amount] * [Measures].[Amount]

SELECT [Square Amount] ON 0 FROM [Sales]
```


 
</div>

           

<div class="ic3-mdx-example-header ic3-mdx-example-header-collapsed" onclick="ic3_mdx_example_click(this)">

#### parameter typing - Value - NULL/EMPTY 

</div>

<div class="ic3-mdx-example-body" style="display:none">

In this example, we are declaring the parameter as a value and calling the function with a NULL and empty tuple. Both are transformed
to the NULL value.

**Query**

```
WITH
  FUNCTION test(Value x_) as IIF(x_ = NULL, 'NULL', 'Something else')
  MEMBER [Empty] AS test( ([Time].[Calendar].[Jan 2010],[Measures].[Amount]) )
  MEMBER [Null] AS test( NULL )
SELECT
 { [Empty],[Null] } on 0
FROM [Sales]
```

 
**Result**

<table  class="ic3-mdx-result">
<tr>
  <td class="ic3-mdx-result-cell-header">Empty</td>
  <td class="ic3-mdx-result-cell-header">Null</td>
</tr>
<tr>
 <td>NULL</td>
 <td>NULL</td>
</tr>
</table>



**Assertion : Cell Equals**

```
WITH
  MEMBER [Empty] AS "NULL"
  MEMBER [Null] AS "NULL"
SELECT
 { [Empty],[Null] } on 0
FROM [Sales]
```


 
</div>

           

<div class="ic3-mdx-example-header ic3-mdx-example-header-collapsed" onclick="ic3_mdx_example_click(this)">

#### return value as any MDX object

</div>

<div class="ic3-mdx-example-body" style="display:none">

In this example, we are returning a member to demonstrate that functions are not limited to scalar value.

**Query**

```
WITH
    FUNCTION f() AS [Measures].defaultMember
SELECT
    f() ON 0,
    [Geography].[Geo].[America].children ON 1
FROM
    [Sales]
```

 
**Result**

<table  class="ic3-mdx-result">
<tr>
  <td class="ic3-mdx-result-cell-empty"></td>
  <td class="ic3-mdx-result-cell-header">Amount</td>
</tr>
<tr>
<td class="ic3-mdx-result-cell-header">&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;Canada</td>
 <td></td>
</tr>
<tr>
<td class="ic3-mdx-result-cell-header">&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;Mexico</td>
 <td></td>
</tr>
<tr>
<td class="ic3-mdx-result-cell-header">&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;United States</td>
 <td>768</td>
</tr>
<tr>
<td class="ic3-mdx-result-cell-header">&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;Venezuela</td>
 <td></td>
</tr>
</table>



**Assertion : Cell Equals**

```
SELECT
    [Measures].defaultMember ON 0,
    [Geography].[Geo].[America].children ON 1
FROM
    [Sales]
```


 
</div>

           

<div class="ic3-mdx-example-header ic3-mdx-example-header-collapsed" onclick="ic3_mdx_example_click(this)">

#### argument value as any MDX object

</div>

<div class="ic3-mdx-example-body" style="display:none">

A function argument may be of any MDX type; in this example we are passing a level as parameter and returning the set of members of this level.

**Query**

```
WITH
    FUNCTION f(lvl) AS lvl.members
SELECT
    [Measures].defaultMember ON 0,
    f( [Geography].[Geo].[Country] ) ON 1
FROM
    [Sales]
```

 
**Result**

<table  class="ic3-mdx-result">
<tr>
  <td class="ic3-mdx-result-cell-empty"></td>
  <td class="ic3-mdx-result-cell-header">Amount</td>
</tr>
<tr>
<td class="ic3-mdx-result-cell-header">&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;Canada</td>
 <td></td>
</tr>
<tr>
<td class="ic3-mdx-result-cell-header">&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;Mexico</td>
 <td></td>
</tr>
<tr>
<td class="ic3-mdx-result-cell-header">&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;United States</td>
 <td>768</td>
</tr>
<tr>
<td class="ic3-mdx-result-cell-header">&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;Venezuela</td>
 <td></td>
</tr>
<tr>
<td class="ic3-mdx-result-cell-header">&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;France</td>
 <td>4</td>
</tr>
<tr>
<td class="ic3-mdx-result-cell-header">&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;Spain</td>
 <td>3</td>
</tr>
<tr>
<td class="ic3-mdx-result-cell-header">&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;Switzerland</td>
 <td>248</td>
</tr>
</table>



**Assertion : Cell Equals**

```
SELECT
    [Measures].defaultMember ON 0,
    [Geography].[Geo].[Country].members ON 1
FROM
    [Sales]
```


 
</div>

           

<div class="ic3-mdx-example-header ic3-mdx-example-header-collapsed" onclick="ic3_mdx_example_click(this)">

#### argument value as a function

</div>

<div class="ic3-mdx-example-body" style="display:none">

A function argument may be a function. In this example, the processor passed as parameter
is applied to the items parameter.

**Query**

```
WITH
    FUNCTION fun(items, proc) AS proc( items )
    MEMBER   val AS fun( [Geography].[Geo].[Country].members, Count )
SELECT
    [val] on 0
FROM
    [Sales]
```

 
**Result**

<table  class="ic3-mdx-result">
<tr>
  <td class="ic3-mdx-result-cell-header">val</td>
</tr>
<tr>
 <td>7</td>
</tr>
</table>



**Assertion : Cell Equals**

```
WITH
    MEMBER   val AS Count( [Geography].[Geo].[Country].members )
SELECT
    [val] on 0
FROM
    [Sales]
```


 
</div>

           

<div class="ic3-mdx-example-header ic3-mdx-example-header-collapsed" onclick="ic3_mdx_example_click(this)">

#### evaluation - current context

</div>

<div class="ic3-mdx-example-body" style="display:none">

A function is always evaluated taking into account the current context. The function defined in this example is returning the value of the default measures for the tuple in the current context of evaluation as defined by the axis and the slicer.

**Query**

```
WITH
    FUNCTION f() AS ( [Geography].[Economy].currentMember, [Measures].[Amount] )
    MEMBER   x   AS f()
SELECT
    x ON 0,
    [Time].[Calendar].[Year].members ON 1
FROM
    [Sales]
WHERE
    [Geography].[Economy].[United States]
```

 
**Result**

<table  class="ic3-mdx-result">
<tr>
  <td class="ic3-mdx-result-cell-empty"></td>
  <td class="ic3-mdx-result-cell-header">x</td>
</tr>
<tr>
<td class="ic3-mdx-result-cell-header">2010</td>
 <td>768</td>
</tr>
<tr>
<td class="ic3-mdx-result-cell-header">2011</td>
 <td></td>
</tr>
</table>



**Assertion : Cell Equals**

```
WITH
    MEMBER [Measures].[x] AS ( [Geography].[Economy].[United States], [Measures].[Amount] )
SELECT
    [Measures].[x] ON 0,
    [Time].[Calendar].[Year].members ON 1
FROM
    [Sales]
```


 
</div>

           

<div class="ic3-mdx-example-header ic3-mdx-example-header-collapsed" onclick="ic3_mdx_example_click(this)">

#### const - current context

</div>

<div class="ic3-mdx-example-body" style="display:none">

In this example, we are defining a function whose evaluation is declared constant. As it cannot see the current context, the constFn() is evaluating the default country (as opposed to the current member) which is the all member.

**Query**

```
WITH
    CONST FUNCTION constFn()    AS ( [Measures].defaultMember, [Geography].[Geo].currentMember )
          FUNCTION nonConstFn() AS ( [Measures].defaultMember, [Geography].[Geo].currentMember )

    MEMBER [Measures].[all-countries]   AS constFn()
    MEMBER [Measures].[current-country] AS nonConstFn()

SELECT
    { [Measures].[all-countries], [Measures].[current-country] } ON 0,
    [Geography].[Geo].[Country].members ON 1

FROM [Sales]
```

 
**Result**

<table  class="ic3-mdx-result">
<tr>
  <td class="ic3-mdx-result-cell-empty"></td>
  <td class="ic3-mdx-result-cell-header">all-countries</td>
  <td class="ic3-mdx-result-cell-header">current-country</td>
</tr>
<tr>
<td class="ic3-mdx-result-cell-header">&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;Canada</td>
 <td>1023</td>
 <td></td>
</tr>
<tr>
<td class="ic3-mdx-result-cell-header">&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;Mexico</td>
 <td>1023</td>
 <td></td>
</tr>
<tr>
<td class="ic3-mdx-result-cell-header">&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;United States</td>
 <td>1023</td>
 <td>768</td>
</tr>
<tr>
<td class="ic3-mdx-result-cell-header">&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;Venezuela</td>
 <td>1023</td>
 <td></td>
</tr>
<tr>
<td class="ic3-mdx-result-cell-header">&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;France</td>
 <td>1023</td>
 <td>4</td>
</tr>
<tr>
<td class="ic3-mdx-result-cell-header">&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;Spain</td>
 <td>1023</td>
 <td>3</td>
</tr>
<tr>
<td class="ic3-mdx-result-cell-header">&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;Switzerland</td>
 <td>1023</td>
 <td>248</td>
</tr>
</table>



**Assertion : Cell Equals**

```
WITH
    MEMBER [Measures].[all-countries]   AS ( [Measures].defaultMember, [Geography].[Geo].[All] )
    MEMBER [Measures].[current-country] AS ( [Measures].defaultMember, [Geography].[Geo].currentMember )

SELECT
    { [Measures].[all-countries], [Measures].[current-country] } ON 0,
    [Geography].[Geo].[Country].members ON 1

FROM [Sales]
```


 
</div>

           

<div class="ic3-mdx-example-header ic3-mdx-example-header-collapsed" onclick="ic3_mdx_example_click(this)">

#### const - slicer visibility and current member

</div>

<div class="ic3-mdx-example-body" style="display:none">

In this example, we are defining a function whose evaluation is declared constant. The slicer is visible from this const function, therefore the .currentMember is [United States] as defined in the slicer.

**Query**

```
WITH
    CONST FUNCTION constFn() AS [Geography].[Geo].currentMember.NAME
    MEMBER [Measures].[const] AS constFn()

SELECT { [Measures].[const] } ON 0 FROM [Sales]

WHERE [Geography].[Geo].[United States]
```

 
**Result**

<table  class="ic3-mdx-result">
<tr>
  <td class="ic3-mdx-result-cell-header">const</td>
</tr>
<tr>
 <td>United States</td>
</tr>
</table>



**Assertion : Cell Equals**

```
WITH MEMBER [Measures].[const] AS [Geography].[Geo].[United States].NAME
SELECT { [Measures].[const] } ON 0 FROM [Sales]
```


 
</div>

           

<div class="ic3-mdx-example-header ic3-mdx-example-header-collapsed" onclick="ic3_mdx_example_click(this)">

#### const - slicer visibility and fact filter

</div>

<div class="ic3-mdx-example-body" style="display:none">

In this example, we are defining a function whose evaluation is declared constant. As the slicer is visible from this const function the default measure for 'All' the regions is equals to the [United States] as specified in the slicer.

**Query**

```
WITH
    CONST FUNCTION constFn() AS ( [Measures].defaultMember, [Geography].[Geo].[All] ).Value
    MEMBER [Measures].[const] AS constFn()

SELECT { [Measures].[const] } ON 0 FROM [Sales]

WHERE [Geography].[Economy].[United States]
```

 
**Result**

<table  class="ic3-mdx-result">
<tr>
  <td class="ic3-mdx-result-cell-header">const</td>
</tr>
<tr>
 <td>768</td>
</tr>
</table>



**Assertion : Cell Equals**

```
WITH MEMBER [Measures].[const] AS ( [Measures].defaultMember, [Geography].[Geo].[United States] ).Value
SELECT { [Measures].[const] } ON 0 FROM [Sales]
```


 
</div>

           

<div class="ic3-mdx-example-header ic3-mdx-example-header-collapsed" onclick="ic3_mdx_example_click(this)">

#### const - sub-select visibility and dimension filter

</div>

<div class="ic3-mdx-example-body" style="display:none">

In this example, we are defining a function whose evaluation is declared constant. As for regular function, the sub-select is taken into account and the [Geography] dimension is filtered.

**Query**

```
WITH
    CONST FUNCTION constFn()    AS IIF( [Geography].[Geo].[France] IS NULL, [Geography].[Geo].[All],           [Geography].[Geo].[Spain] )
          FUNCTION nonConstFn() AS IIF( [Geography].[Geo].[France] IS NULL, [Geography].[Geo].[United States], [Geography].[Geo].[Spain] )

SELECT { constFn(), nonConstFn() } ON 0

FROM (SELECT [Geography].[Geo].[United States] ON 0 FROM [Sales] )
```

 
**Result**

<table  class="ic3-mdx-result">
<tr>
  <td class="ic3-mdx-result-cell-header">All Regions</td>
  <td class="ic3-mdx-result-cell-header">United States</td>
</tr>
<tr>
 <td>768</td>
 <td>768</td>
</tr>
</table>



**Assertion : Cell Equals**

```
SELECT { [Geography].[Geo].[All], [Geography].[Geo].[United States] } ON 0
FROM (SELECT [Geography].[Geo].[United States] ON 0 FROM [Sales] )
```


 
</div>

           

<div class="ic3-mdx-example-header ic3-mdx-example-header-collapsed" onclick="ic3_mdx_example_click(this)">

#### const - sub-select visibility and fact filter

</div>

<div class="ic3-mdx-example-body" style="display:none">

In this example, we are defining a function whose evaluation is declared constant. As for regular function, the sub-select is taken into account and the value of all the regions is filtered by the sub-select specifying [United States] only.

**Query**

```
WITH
    CONST FUNCTION constFn()    AS ( [Geography].[Geo].[All], [Measures].defaultMember ).value
          FUNCTION nonConstFn() AS ( [Geography].[Geo].[All], [Measures].defaultMember ).value

    MEMBER [const-fn]     AS constFn()
    MEMBER [non-const-fn] AS nonConstFn()

SELECT { [const-fn], [non-const-fn] } ON 0
FROM (SELECT [Geography].[Geo].[United States] ON 0 FROM [Sales] )
```

 
**Result**

<table  class="ic3-mdx-result">
<tr>
  <td class="ic3-mdx-result-cell-header">const-fn</td>
  <td class="ic3-mdx-result-cell-header">non-const-fn</td>
</tr>
<tr>
 <td>1023</td>
 <td>1023</td>
</tr>
</table>



**Assertion : Cell Equals**

```
WITH
    MEMBER [const-fn]     AS ( [Geography].[Geo].[All], [Measures].defaultMember ).value
    MEMBER [non-const-fn] AS ( [Geography].[Geo].[All], [Measures].defaultMember ).value

SELECT { [const-fn], [non-const-fn] } ON 0
FROM (SELECT [Geography].[Geo].[United States] ON 0 FROM [Sales] )
```


 
</div>

           

<div class="ic3-mdx-example-header ic3-mdx-example-header-collapsed" onclick="ic3_mdx_example_click(this)">

#### const - more on current context visibility

</div>

<div class="ic3-mdx-example-body" style="display:none">

The current context of a const function is made of the slicer and sub-select only; it cannot see for example the current tuple of the cell being evaluated. Therefore, in this example, the evaluation of the function is done using the calendar default value (i.e., [2010]); the days on the column axis are ignored.

**Query**

```
WITH
    CONST FUNCTION f() AS ( [Measures].[Amount] ).Value
    MEMBER ConstFunctionValue AS f()
SELECT
   { [Measures].[Amount], ConstFunctionValue } ON 0,
   { [Time].[Calendar].defaultMember, [Time].[Calendar].[May 2010].children } on 1
FROM [Sales]
WHERE [Product].[Company].&[Crazy Development].[icCube].[Corporate]
```

 
**Result**

<table  class="ic3-mdx-result">
<tr>
  <td class="ic3-mdx-result-cell-empty"></td>
  <td class="ic3-mdx-result-cell-header">Amount</td>
  <td class="ic3-mdx-result-cell-header">ConstFunctionValue</td>
</tr>
<tr>
<td class="ic3-mdx-result-cell-header">2010</td>
 <td>912</td>
 <td>912</td>
</tr>
<tr>
<td class="ic3-mdx-result-cell-header">&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;May 1 2010</td>
 <td></td>
 <td>912</td>
</tr>
<tr>
<td class="ic3-mdx-result-cell-header">&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;May 2 2010</td>
 <td></td>
 <td>912</td>
</tr>
<tr>
<td class="ic3-mdx-result-cell-header">&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;May 3 2010</td>
 <td></td>
 <td>912</td>
</tr>
<tr>
<td class="ic3-mdx-result-cell-header">&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;May 4 2010</td>
 <td></td>
 <td>912</td>
</tr>
<tr>
<td class="ic3-mdx-result-cell-header">&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;May 5 2010</td>
 <td></td>
 <td>912</td>
</tr>
<tr>
<td class="ic3-mdx-result-cell-header">&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;May 6 2010</td>
 <td></td>
 <td>912</td>
</tr>
<tr>
<td class="ic3-mdx-result-cell-header">&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;May 7 2010</td>
 <td></td>
 <td>912</td>
</tr>
<tr>
<td class="ic3-mdx-result-cell-header">&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;May 8 2010</td>
 <td></td>
 <td>912</td>
</tr>
<tr>
<td class="ic3-mdx-result-cell-header">&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;May 9 2010</td>
 <td></td>
 <td>912</td>
</tr>
<tr>
<td class="ic3-mdx-result-cell-header">&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;May 10 2010</td>
 <td></td>
 <td>912</td>
</tr>
<tr>
<td class="ic3-mdx-result-cell-header">&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;May 11 2010</td>
 <td></td>
 <td>912</td>
</tr>
<tr>
<td class="ic3-mdx-result-cell-header">&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;May 12 2010</td>
 <td></td>
 <td>912</td>
</tr>
<tr>
<td class="ic3-mdx-result-cell-header">&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;May 13 2010</td>
 <td></td>
 <td>912</td>
</tr>
<tr>
<td class="ic3-mdx-result-cell-header">&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;May 14 2010</td>
 <td></td>
 <td>912</td>
</tr>
<tr>
<td class="ic3-mdx-result-cell-header">&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;May 15 2010</td>
 <td></td>
 <td>912</td>
</tr>
<tr>
<td class="ic3-mdx-result-cell-header">&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;May 16 2010</td>
 <td></td>
 <td>912</td>
</tr>
<tr>
<td class="ic3-mdx-result-cell-header">&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;May 17 2010</td>
 <td>16</td>
 <td>912</td>
</tr>
<tr>
<td class="ic3-mdx-result-cell-header">&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;May 18 2010</td>
 <td></td>
 <td>912</td>
</tr>
<tr>
<td class="ic3-mdx-result-cell-header">&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;May 19 2010</td>
 <td></td>
 <td>912</td>
</tr>
<tr>
<td class="ic3-mdx-result-cell-header">&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;May 20 2010</td>
 <td></td>
 <td>912</td>
</tr>
<tr>
<td class="ic3-mdx-result-cell-header">&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;May 21 2010</td>
 <td></td>
 <td>912</td>
</tr>
<tr>
<td class="ic3-mdx-result-cell-header">&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;May 22 2010</td>
 <td>128</td>
 <td>912</td>
</tr>
<tr>
<td class="ic3-mdx-result-cell-header">&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;May 23 2010</td>
 <td></td>
 <td>912</td>
</tr>
<tr>
<td class="ic3-mdx-result-cell-header">&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;May 24 2010</td>
 <td></td>
 <td>912</td>
</tr>
<tr>
<td class="ic3-mdx-result-cell-header">&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;May 25 2010</td>
 <td>256</td>
 <td>912</td>
</tr>
<tr>
<td class="ic3-mdx-result-cell-header">&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;May 26 2010</td>
 <td></td>
 <td>912</td>
</tr>
<tr>
<td class="ic3-mdx-result-cell-header">&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;May 27 2010</td>
 <td></td>
 <td>912</td>
</tr>
<tr>
<td class="ic3-mdx-result-cell-header">&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;May 28 2010</td>
 <td></td>
 <td>912</td>
</tr>
<tr>
<td class="ic3-mdx-result-cell-header">&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;May 29 2010</td>
 <td>512</td>
 <td>912</td>
</tr>
<tr>
<td class="ic3-mdx-result-cell-header">&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;May 30 2010</td>
 <td></td>
 <td>912</td>
</tr>
<tr>
<td class="ic3-mdx-result-cell-header">&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;May 31 2010</td>
 <td></td>
 <td>912</td>
</tr>
</table>



**Assertion : Cell Equals**

```
WITH
    MEMBER ConstFunctionValue AS ( [Time].[Calendar].defaultMember, [Measures].[Amount] )
SELECT
    { [Measures].[Amount], ConstFunctionValue } ON 0,
    { [Time].[Calendar].defaultMember, [Time].[Calendar].[May 2010].children } on 1
FROM [Sales]
WHERE [Product].[Company].&[Crazy Development].[icCube].[Corporate]
```


 
</div>

                                                                                                            

### See Also


[Create (Function)](Create%20(Function).md)

[Drop (Function)](Drop%20(Function).md)


### XMLA Properties

Find [here](icCubeDefaultXmlaProperties.md) the value of the XMLA properties when not specified otherwise.

_
